/**

VIDEO ANALYSIS PREPROCESSING  - BASE CLASSES
author: Pavel Zak, izakpa@fit.vutbr.cz

*/

#include "preprocessing.h"

#include "highgui.h"

#include <algorithm>


using namespace std;

int clip(int val){
	return min(max(val, 0),255);
}

void YUV4442RGB(UINT8 Y, UINT8 U, UINT8 V, UINT8 &R, UINT8 &G, UINT8 &B){
	int C = Y-16;
	int D = U-128;
	int E = V-128;
	R = clip((298*C + 409*E + 128)>>8);
	G = clip((298*C -100*D - 208*E + 128)>>8);
	B = clip((298*C + 516*D + 128)>>8);
}

void YUV4202RGB(UINT8 * p_data, int x, int y, int width, int height, UINT8 &R, UINT8 &G, UINT8 &B){
	int total = width * height;
	UINT8 Y = p_data[y * width + x];
	//UINT8 U = p_data[y * (width / 4) + (x / 2) + total];
	//UINT8 V = p_data[y * (width / 4) + (x / 2) + total + (total / 4)];

	UINT8 U = p_data[(y / 2)* (width / 2) + (x / 2) + total];
	UINT8 V = p_data[(y / 2) * (width / 2) + (x / 2) + total + (total / 4)];

	
	YUV4442RGB(Y, U, V, R, G, B);
}

int CreateIplFromYUVbuffer(UINT8 * p_data, IplImage ** image, int width, int height){
	UINT8 r, g, b;
	int index;

#ifdef USE_IPP
	const Ipp8u* pSrc[3];
	Ipp8u* pDst = NULL;
	IppiSize size;
	size.width = width;
	size.height = height;

	pSrc[0] = p_data;
	pSrc[1] = p_data+(width*height)+(width*height)/4;
	pSrc[2] = p_data+(width*height);//+(width*height)/4;
#endif

	if ((*image)!=NULL){
		if (((*image)->width!=width)||((*image)->height!=height)||((*image)->nChannels!=3)||((*image)->depth!=8)){
			cvReleaseImage(image);
			*image = cvCreateImage(cvSize(width, height), 8, 3);
		}
	}
	else{
		*image = cvCreateImage(cvSize(width, height), 8, 3);
	}
#ifdef USE_IPP
	ippiYUV420ToRGB_8u_P3C3(pSrc, (Ipp8u*)((*image)->imageData), size);
#else

	for (int y=0; y<height; y++){
		for (int x=0; x<width; x++){
			YUV4202RGB(p_data, x, y, width, height, r, g, b);
			index = 3*(y*width + x);
			(*image)->imageData[index] = b;
			(*image)->imageData[index+1] = g;
			(*image)->imageData[index+2] = r;
		}
	}
#endif
	return 0;
}

int CreateGrayIplFromYUVbuffer(UINT8 * p_data, IplImage ** image, int width, int height){
	if ((*image)!=NULL){
		if (((*image)->width!=width)||((*image)->height!=height)||((*image)->nChannels!=3)||((*image)->depth!=8)){
			cvReleaseImage(image);
			*image = cvCreateImage(cvSize(width, height), 8, 1);
		}
	}
	else{
		*image = cvCreateImage(cvSize(width, height), 8, 1);
	}
	
	//copy Y values
	memcpy((*image)->imageData, p_data, sizeof(UINT8)*width*height);

	return 0;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Preproc_image::Preproc_image(string source){
	image = NULL;
	sourceID = source;
	frameNo = 0;
}

Preproc_image::Preproc_image(){
	image = NULL;
	sourceID = "";
	frameNo = 0;
}

Preproc_image::Preproc_image(const Preproc_image &preproc){
	//cvCopyImage(preproc.image, image);
	if (preproc.image!=NULL)
		image = cvCloneImage(preproc.image);
	else image = NULL;
	//image = preproc.image;
	sourceID = preproc.sourceID;
	frameNo = preproc.frameNo;
}

Preproc_image & Preproc_image::operator= (const Preproc_image & preproc){
        if (this != &preproc){ // protect against invalid self-assignment
					if (image) cvReleaseImage(&image);
					if (preproc.image!=NULL)
						image = cvCloneImage(preproc.image);
					else image = NULL;
					//image = preproc.image;
					sourceID = preproc.sourceID;
					frameNo = preproc.frameNo;
        }
        return *this;
}


Preproc_image::~Preproc_image(){
	if (image) cvReleaseImage(&image);
	image = NULL;
}

void Preproc_image::update(IplImage * act_image, int64 timestamp){
	if (image!=NULL)
		cvReleaseImage(&image);

	image = act_image;
	frameNo = timestamp;
}

IplImage * Preproc_image::getImage(){
	return image;
}

string Preproc_image::getSource(){
	return sourceID;
}

int64 Preproc_image::getTimestamp(){
	return frameNo;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////



Preprocessing::Preprocessing(){
	sm = new TSMObject( 50, TSMObject::defaultSMName, true, 100);
	//sm = ssm;
	proc_trees.clear();
	queues.clear();
	terminals.clear();
	image_sources.clear();
	//cout << "vel" << image_sources.size() << endl;
}

void Preprocessing::register_queue(string queue){
	sm->registerQueue(queue);
}


Preprocessing::~Preprocessing(){
	delete sm;
	//sm = ssm;
	proc_trees.clear();
	queues.clear();
	terminals.clear();
	image_sources.clear();
}

	
int Preprocessing::processSM(){

	map< string, bool> queues;

	//vycteni identifikatoru ze SM
	sm->getActiveQueues(queues);
	//sestaveni preproc stromu

	buildTree(queues);

	//zpracovani preproc stromu
	processImages();

	//zpet do SM
	uploadImages();
	
	return 0;
}

int Preprocessing::processImage(IplImage * image, int64 frameNo, string sourceID){
	for (unsigned int i=0; i<proc_trees.size(); i++){
		proc_trees[i]->process(image, frameNo, sourceID);
	}
	return 0;
}


#define STATE_ERR 0
#define STATE_SOURCE 1
#define STATE_SOURCE_DONE 2
#define STATE_OPERATION 3
#define STATE_PARAMETER_NAME 4
#define STATE_PARAMETER_VALUE 5

int Preprocessing::buildProcQueue(string qid){
		typedef boost::tokenizer<boost::char_separator<char> > tokenizer;
		string act_token;
		boost::char_separator<char> sep("_");
    tokenizer tokens(qid, sep);
		int state = STATE_SOURCE;

		Preproc_image * im = NULL;
		map<string, Preproc_image>::iterator it;

		string source = "";
		string operation = "";
		string parameter = "";
		string value = "";

		map<string, string> parameters;
		parameters.clear();
		Preproc_module * mod = NULL;
		Preproc_module * act_module = NULL;
		Preproc_module * root_module = NULL;
		
    for (tokenizer::iterator tok_iter = tokens.begin(); tok_iter != tokens.end(); ++tok_iter){
			act_token = *tok_iter;
      
			switch (state){
				case STATE_SOURCE:
					source = act_token;
					state = STATE_SOURCE_DONE;
					operation = "SOURCE";
					parameters[operation] = source;
					//pridani zdroje
					it = image_sources.find(source);
					if (it==image_sources.end()){
						im = new Preproc_image(source);
						image_sources[source] = *im;
						im = NULL;
					}
					break;

				case STATE_SOURCE_DONE:
					if (act_token=="!") state = STATE_OPERATION;
					else state = STATE_ERR;
					break;

				case STATE_OPERATION:
					mod = factory.getPreproc_Module(operation, parameters, qid);
					parameters.clear();
					if ((mod!=NULL)&&(mod->isOK())){
						if (!act_module){
							root_module = act_module = mod;
							mod = NULL;
						}
						else {
							act_module->add_next(mod);
							act_module = mod;
							mod = NULL;
						}
					}
					else {
						state = STATE_ERR;
						delete mod;
						mod = NULL;
					}
			
					if (act_token=="!") state = STATE_ERR;
					else {
						operation = act_token;
						state = STATE_PARAMETER_NAME;
					}
					break;
				case STATE_PARAMETER_NAME:
					if (act_token=="!") state = STATE_OPERATION;
					else {
						parameter = act_token;
						state = STATE_PARAMETER_VALUE;
					}
					break;
				case STATE_PARAMETER_VALUE:
					if (act_token=="!") state = STATE_OPERATION;
					else {
						value = act_token;
						state = STATE_PARAMETER_NAME;
						parameters[parameter]=value;
					}
					break;
				case STATE_ERR:
						
					break;
			}//switch
		}//for
		if (state!=STATE_ERR){//parsing OK
			if (state==STATE_OPERATION){
				mod = factory.getPreproc_Module(operation, parameters, qid);
					parameters.clear();
					if ((mod!=NULL)&&(mod->isOK())){
						if (!act_module){
							root_module = act_module = mod;
							mod = NULL;
						}
						else {
							act_module->add_next(mod);
							act_module = mod;
							mod = NULL;
						}
					}
					else{
						delete mod;
						return 0;
					}
			}
			proc_trees.push_back(root_module);
			terminals.push_back(act_module);
		}
		else {//parsing fail, clear all allocated memory
			delete root_module;
			return 0;
		}
	return 1;
}

void Preprocessing::showResults(){
	for (unsigned int i=0; i<terminals.size(); i++){
		cvNamedWindow(terminals[i]->getQid().c_str());
		cvShowImage(terminals[i]->getQid().c_str(), terminals[i]->getResult());
	}
}


int Preprocessing::buildTree(map< string, bool> aqueues){
	map< string, bool>::iterator it;
	bool is_ok = true;
	vector<string>::iterator found;
	for (it = aqueues.begin(); it != aqueues.end(); it++){
		
		found = find(queues.begin(),queues.end(), it->first);
		if ((it->second)&&(found==queues.end())){
			is_ok &= (buildProcQueue(it->first)==1);
			if (is_ok) queues.push_back(it->first);
		}
	}
	return is_ok;
}

int Preprocessing::processImages(){
	map<string, Preproc_image>::iterator it;
	for (it=image_sources.begin(); it!=image_sources.end(); it++){
		processImage(it->second.getImage(), it->second.getTimestamp(), it->first);
	}
	return 0;
}

int  Preprocessing::uploadImages(){
	IplImage * result;
	string qid;
	int64 ts;

	for (unsigned int i=0; i<terminals.size(); i++){
		//cvNamedWindow();
		//cvShowImage(terminals[i]->getQid().c_str(), );
		result = terminals[i]->getResult();
		qid = terminals[i]->getQid();
		ts = terminals[i]->getTimeStamp();

		if (result!=NULL){
		  IplImage *smImage = sm->createIplImage( cvGetSize( result), result->depth, result->nChannels);
		  cvCopy( result, smImage);
		  sm->pushImage( qid, smImage, ts);
		  sm->cleanup();
		}
		else {
			cerr << "Preprocessing: NULL pointer image" << endl;
		}
	}
	return 0;
}

int Preprocessing::addSourceFrame(IplImage * image, string sourceID, int64 timestamp){
	map<string, Preproc_image>::iterator it = image_sources.end();
	Preproc_image * n = NULL;
	
	it = image_sources.find(sourceID);

	if (it!=image_sources.end()){//nalezeno
		it->second.update(image, timestamp);	
	}
	else {
		n = new Preproc_image(sourceID);
		n->update(image, timestamp);
		image_sources[sourceID] = *n;
		n = NULL;
	}
	
	return 0;
}

int Preprocessing::addSourceFrameYUV(UINT8 * p_data, int width, int height, string sourceID, int64 timestamp){
	IplImage * image = NULL;
	IplImage * gray = NULL;
	string grayID = sourceID+"-GRAY";

	CreateIplFromYUVbuffer(p_data, &image, width, height);
	CreateGrayIplFromYUVbuffer(p_data, &gray, width, height);

	addSourceFrame(image, sourceID, timestamp);
	addSourceFrame(gray, grayID, timestamp);
	
	return 0;
}
